<?php

namespace App\Http\Controllers;

use App\Banner;
use Illuminate\Http\Request;
use App\Http\Requests\ValidationRequest;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Helper;
use File;
use Image;
use Alert;
use Illuminate\Database\QueryException;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class BannerController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        return view('pages.be.banners');
    }

    function json()
    {
        set_time_limit(0);
        ini_set('memory_limit', '-1');
        $data = Banner::orderBy('id', 'DESC')->get();
        return DataTables::of($data)
                    ->addIndexColumn()
                    ->editColumn('options', function($row){
                        $count = Banner::count();
                        if($count <= 1)
                        {
                            $disabled = "disabled";
                        }else{
                            $disabled = "";
                        }

                        $edit = route('banner.edit', $row->id);
                        $btn = "&emsp;<a href='{$edit}' class='btn btn-sm btn-warning modal-show edit' title='Form Edit Banner: {$row->name}'><i class='fa fa-pencil'></i></a> <button type='button' class='btn btn-sm btn-danger' id='destroy' {$disabled} data-id='{$row->id}'><i class='fa fa-trash'></i></button>";
                        return $btn;
                    })
                    ->editColumn('picture', function($row){
                        if($row->picture != NULL){
                            $picture = "<img src=".url('assets/files/banner').'/'.$row->picture." width='100px'>";
                        }else{
                            $picture = "<img src=".url('assets/frontend/img/no-picture.png')." width='100px'>";
                        }
                        return $picture;
                    })
                    ->editColumn('status', function($row){
                        if($row->status == 1){
                            $status = "<label class='switch switch-primary'><input data-id='{$row->id}' type='checkbox' id='status' checked ><span></span></label>";
                        }else{
                            $status = "<label class='switch switch-primary'><input data-id='{$row->id}' type='checkbox' id='status' ><span></span></label>";
                        }
                        return $status;
                    })
                    ->editColumn('created_at', function($row){
                        return $row->created_at != NULL ? \Helper::tgl_indo($row->created_at) : "-";
                    })
                    ->editColumn('updated_at', function($row){
                        return $row->updated_at != NULL ? \Helper::tgl_indo($row->updated_at) : "-";
                    })
                    ->rawColumns(['options', 'picture', 'status'])
                    ->toJson();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $model = new Banner();
        return view('pages.be.form_banner', compact('model'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $this->validate($request, [
                'name' => 'required|string',
                'description' => 'required'
            ],[
                'name.required' => 'Nama Banner harus diisi!',
                'name.string' => 'Nama Banner hanya string!',
                'description.required' => 'Deskripsi harus diisi!'
            ]);

            $data = new Banner();
            $data->name = $request->name;
            $data->description = $request->description;
            $data->slug = Helper::createBannerSlug($request->name);
            $data->created_at = Carbon::now('Asia/Jakarta');
            $data->updated_at = null;
            $data->save();
            \Helper::log(\Auth::user()->username, "Menambahkan banner slider : ".$data->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menambah banner slider!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Banner  $banner
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $model = Banner::findOrFail($id);
        return view('pages.be.form_banner', compact('model'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Banner  $banner
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $this->validate($request, [
                'name' => 'required|string',
                'description' => 'required',
                'picture' => 'file|max:500|image|mimes:jpg,jpeg,png,JPG,JPEG,PNG'
            ],[
                'name.required' => 'Nama Banner harus diisi!',
                'name.string' => 'Nama Banner hanya string!',
                'description.required' => 'Deskripsi harus diisi!',
                'picture.image' => 'Hanya boleh gambar diupload!',
                'picture.file' => 'Hanya tipe data file!'
            ]);
            $banner = Banner::findOrFail($id);
            if(!empty($request->file('picture'))){
                if(file_exists("assets/files/banner/".$banner->picture)){
                    File::delete('assets/files/banner/'.$banner->picture);
                    if(file_exists("assets/files/banner/thumbnail/".$banner->picture)){
                        File::delete('assets/files/banner/thumbnail/'.$banner->picture);
                    }
                }
                if(!File::isDirectory('assets/files/banner')){
                    File::makeDirectory('assets/files/banner', 0755, true, true);
                }
                if(!File::isDirectory('assets/files/banner/thumbnail')) {
                    File::makeDirectory('assets/files/banner/thumbnail', 0755, true, true);
                }
                $file = $request->file('picture');
                $filename = strtolower('banner-'.Helper::acakstring(8)).'.'.$request->picture->extension();
                $path = "assets/files/banner/".$filename;
                $uploaded = Helper::resizeImage($file, $path, 1200, 580);
                if($uploaded){
                    $thumbnailsource = "assets/files/banner/".$filename;
                    $thumbnailpath = "assets/files/banner/thumbnail/". $filename;
                    Helper::createThumbnail($thumbnailsource, $thumbnailpath, 600, 290);
                }
                //create backupdata
                // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/banner/", $path, $filename);
                // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/banner/thumbnail", $thumbnailpath, $filename);
                $data = array(
                    'name' => $request->name,
                    'slug' => Helper::createBannerSlug($request->name, $id),
                    'description' => $request->description,
                    'picture' => $filename,
                    'updated_at' => Carbon::now('Asia/Jakarta')
                );
            }else{
                $data = array(
                    'name' => $request->name,
                    'slug' => Helper::createBannerSlug($request->name, $id),
                    'description' => $request->description,
                    'updated_at' => Carbon::now('Asia/Jakarta')
                );
            }
            $banner->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate banner slider : ".$data['name']);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengupdate banner slider!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function update_status(Request $request){
        try {
            $id = $request->id;
            $banner = Banner::findOrFail($id);
            //proteksi banner
            $cek = Banner::where('status', '1')->count();
            if($banner->status == 0){
                $proteksi = 0;
                $data = array('status' => '1', 'updated_at' => Carbon::now('Asia/Jakarta'));
            }else{
                $proteksi = 1;
                $data = array('status' => '0', 'updated_at' => Carbon::now('Asia/Jakarta'));
            }
            if($cek == 1 && $proteksi == 1){
                return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => 'Gagal mengupdate status banner, minimal harus ada 1 banner aktif!']);
            }else{
                $banner->update($data);
                \Helper::log(\Auth::user()->username, "Mengupdate status banner slider : ".$banner->name);
                return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengupdate status banner!']);
            }
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }

    }

    public function update_gambar(ValidationRequest $request)
    {
        try {
            $this->validate($request, [
                'picture' => 'required'
            ],[
            'picture.required' => 'Gambar tidak boleh kosong!'
            ]);
            $id = $request->id;
            $banner = Banner::findOrFail($id);
            if(file_exists("assets/files/banner/".$banner->picture)){
                File::delete('assets/files/banner/'.$banner->picture);
                if(file_exists("assets/files/banner/thumbnail/".$banner->picture)){
                    File::delete('assets/files/banner/thumbnail/'.$banner->picture);
                }
            }
            if(!File::isDirectory('assets/files/banner')){
                File::makeDirectory('assets/files/banner', 0755, true, true);
            }
            if(!File::isDirectory('assets/files/banner/thumbnail')) {
                File::makeDirectory('assets/files/banner/thumbnail', 0755, true, true);
            }

            $file = $request->file('picture');
            $filename = strtolower('banner-'.Helper::acakstring(8)).'.'.$request->picture->extension();
            $path = "assets/files/banner/".$filename;
            $uploaded = Helper::resizeImage($file, $path, 1200, 580);
            if($uploaded){
                $thumbnailsource = "assets/files/banner/".$filename;
                $thumbnailpath = "assets/files/banner/thumbnail/". $filename;
                Helper::createThumbnail($thumbnailsource, $thumbnailpath, 600, 290);
            }
            //create backupdata
            // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/banner/", $path, $filename);
            // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/banner/thumbnail", $thumbnailpath, $filename);
            $data = array('picture' => $filename, 'updated_at' => Carbon::now('Asia/Jakarta'));
            $banner->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate gambar banner slider : ".$banner->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengupdate gambar banner!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function hapus_gambar(Request $request)
    {
        try {
            $id = $request->id;
            $banner = Banner::findOrFail($id);
            if(file_exists("assets/files/banner/".$banner->picture)){
                File::delete('assets/files/banner/'.$banner->picture);
                if(file_exists("assets/files/banner/thumbnail/".$banner->picture)){
                    File::delete('assets/files/banner/thumbnail/'.$banner->picture);
                }
            }
            $banner->update(['picture' => null, 'updated_at' => Carbon::now('Asia/Jakarta')]);
            \Helper::log(\Auth::user()->username, "Menghapus gambar banner slider : ".$banner->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus gambar banner!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Banner  $banner
     * @return \Illuminate\Http\Response
     */
    public function destroy(Banner $banner)
    {
        //
    }

    public function remove(Request $request)
    {
        try {
            $id = $request->input('id');
            $banner = Banner::findOrFail($id);
            if(file_exists("assets/files/banner/".$banner->picture)){
                File::delete('assets/files/banner/'.$banner->picture);
                if(file_exists("assets/files/banner/thumbnail/".$banner->picture)){
                    File::delete('assets/files/banner/thumbnail/'.$banner->picture);
                }
            }
            $banner->delete();
            \Helper::log(\Auth::user()->username, "Menghapus banner slider : ".$banner->name);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus banner!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }
}
